#ifndef __CCriticalSectionScope__
#define __CCriticalSectionScope__

#include "CCriticalSection.hpp"
using Exponent::Threading::CCriticalSection;

//	===========================================================================

namespace Exponent
{
	namespace Threading
	{
		/**
		 * @class CCriticalSectionScope CCriticalSectionScope.hpp
		 * @brief A wrapper around a cross platform critical section that is scope specific and requires a critical section to be supplied
		 *
		 * This class works like an automatic pointer, but with specifics for critical sections. In the future it will be made an alternative object deleter\n
		 * that can be passed to the automatic pointer. Be aware that this update will happen in the future, although this class will not be deprecated at that point.\n
		 * To use this class, you should create it as a local variable. This enters the critical section. When the scope is deleted it will also leave the critical section
		 * @code
		 * void foo(double *array, const long size)
		 * {
		 *		// We enter the critical section wiht this call
		 *      CCriticalSectionScope scope(new CCriticalSection);
		 *
		 *      // Do our critical thing
		 *		array[0] = 1.0;
		 *     
		 *		// scope is automatically destroyed here and takes out the critical section with it...
		 *		// you do not need to manually exit the critical section
		 * }
		 * @endcode
		 *
		 * @date 10/08/2005
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: CCriticalSectionScope.hpp,v 1.6 2007/02/08 21:06:44 paul Exp $
		 */
		class CCriticalSectionScope
		{
		public:

			/**
			 * Construction
			 * @param criticalSection The critical section to use
			 */
			CCriticalSectionScope(CCriticalSection &criticalSection);

			/**
			 * Construction
			 * @param criticalSection The critical section to use
			 */
			CCriticalSectionScope(CCriticalSection *criticalSection = NULL);

			/**
			 * Construction
			 * @param criticalSection The critical section to use
			 * @param waitTimeInMilliseconds The amount of time to wait between attempts to enter the critical section
			 */
			CCriticalSectionScope(CCriticalSection &criticalSection, const long waitTimeInMilliseconds);

			/**
			 * Construction
			 * @param criticalSection The critical section to use
			 * @param waitTimeInMilliseconds The amount of time to wait between attempts to enter the critical section
			 */
			CCriticalSectionScope(CCriticalSection *criticalSection, const long waitTimeInMilliseconds);

			/**
			 * Destruction
			 */
			~CCriticalSectionScope();

//	===========================================================================

			/**
			 * Set the ciritical section and enter it
			 * @param criticalSection The critical section to use
			 */
			void setAndEnterCriticalSection(CCriticalSection *criticalSection);

			/**
			 * Set the critical section and try to enter it
			 * @param criticalSection The critical section to use
			 * @param waitTimeInMilliseconds The amount of time to wait between attempts to enter the critical section
			 * @retval bool True if we entered the critical section sucessfuly, false otherwise or error
			 */
			bool setAndTryToEnterCriticalSection(CCriticalSection *criticalSection, const long waitTimeInMilliseconds);

			/**
			 * End the critical section
			 */
			void endCriticalSection();

//	===========================================================================

			/**
			 * Are we inside the critical section
			 * @retval bool True if inside critical section, false otherwise
			 */
			bool hasEntered() const { return m_entered; }

//	===========================================================================

		protected:

//	===========================================================================

			bool m_entered;								/**< Have we entered the critical section? */
			CCriticalSection *m_criticalSection;		/**< The ciritical section that we are working with */
		};
	}
}


#endif	// End of CCriticalSectionScope.hpp